/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.biome.v1;

import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.function.BiPredicate;
import net.minecraft.class_12197;
import net.minecraft.class_12199;
import net.minecraft.class_12206;
import net.minecraft.class_12212;
import net.minecraft.class_1299;
import net.minecraft.class_1311;
import net.minecraft.class_1959;
import net.minecraft.class_2893;
import net.minecraft.class_2922;
import net.minecraft.class_4763;
import net.minecraft.class_5321;
import net.minecraft.class_5483;
import net.minecraft.class_6010;
import net.minecraft.class_6796;
import org.jetbrains.annotations.UnmodifiableView;

/**
 * Allows {@link class_1959} properties to be modified.
 */
public interface BiomeModificationContext {
	/**
	 * Returns the modification context for the biomes weather properties.
	 */
	WeatherContext getWeather();

	/**
	 * Returns the modification context for the biomes environment attributes.
	 */
	AttributesContext getAttributes();

	/**
	 * Returns the modification context for the biomes effects.
	 */
	EffectsContext getEffects();

	/**
	 * Returns the modification context for the biomes generation settings.
	 */
	GenerationSettingsContext getGenerationSettings();

	/**
	 * Returns the modification context for the biomes spawn settings.
	 */
	SpawnSettingsContext getSpawnSettings();

	interface WeatherContext {
		/**
		 * @see class_1959#method_48163()
		 * @see class_1959.class_1960#method_48164(boolean)
		 */
		void setPrecipitation(boolean hasPrecipitation);

		/**
		 * @see class_1959#method_8712()
		 * @see class_1959.class_1960#method_8747(float)
		 */
		void setTemperature(float temperature);

		/**
		 * @see class_1959.class_1960#method_30777(class_1959.class_5484)
		 */
		void setTemperatureModifier(class_1959.class_5484 temperatureModifier);

		/**
		 * @see class_1959.class_5482#comp_846()
		 * @see class_1959.class_1960#method_8727(float)
		 */
		void setDownfall(float downfall);
	}

	interface AttributesContext {
		/**
		 * @see class_1959.class_1960#method_75739(class_12199)
		 */
		void addAll(class_12199 map);

		/**
		 * @see class_1959.class_1960#method_75738(class_12199.class_12200)
		 */
		default void addAll(class_12199.class_12200 map) {
			this.addAll(map.method_75672());
		}

		/**
		 * @see class_1959.class_1960#method_75737(class_12197, Object)
		 */
		<T> void set(class_12197<T> key, T value);

		/**
		 * @see class_1959.class_1960#method_75736(class_12197, class_12212, Object)
		 */
		<T, M> void setModifier(class_12197<T> key, class_12212<T, M> modifier, M value);
	}

	interface EffectsContext {
		/**
		 * @deprecated Set the fog color using environment attributes instead
		 * @see BiomeModificationContext#getAttributes()
		 * @see class_12206#field_63742
		 */
		@Deprecated
		void setFogColor(int color);

		/**
		 * @see class_4763#getWaterColor()
		 * @see class_4763.class_4764#method_24395(int)
		 */
		void setWaterColor(int color);

		/**
		 * @deprecated Set the water fog color using environment attributes instead
		 * @see BiomeModificationContext#getAttributes()
		 * @see class_12206#field_63744
		 */
		@Deprecated
		void setWaterFogColor(int color);

		/**
		 * @deprecated Set the sky color using environment attributes instead
		 * @see BiomeModificationContext#getAttributes()
		 * @see class_12206#field_63746
		 */
		@Deprecated
		void setSkyColor(int color);

		/**
		 * @see class_4763#getFoliageColor()
		 * @see class_4763.class_4764#method_30821(int)
		 */
		void setFoliageColor(Optional<Integer> color);

		/**
		 * @see class_4763#getFoliageColor()
		 * @see class_4763.class_4764#method_30821(int)
		 */
		default void setFoliageColor(int color) {
			setFoliageColor(Optional.of(color));
		}

		/**
		 * @see class_4763#getFoliageColor()
		 * @see class_4763.class_4764#method_30821(int)
		 */
		default void setFoliageColor(OptionalInt color) {
			color.ifPresentOrElse(this::setFoliageColor, this::clearFoliageColor);
		}

		/**
		 * @see class_4763#getFoliageColor()
		 * @see class_4763.class_4764#method_30821(int)
		 */
		default void clearFoliageColor() {
			setFoliageColor(Optional.empty());
		}

		/**
		 * @see class_4763#getGrassColor()
		 * @see class_4763.class_4764#method_30822(int)
		 */
		void setGrassColor(Optional<Integer> color);

		/**
		 * @see class_4763#getGrassColor()
		 * @see class_4763.class_4764#method_30822(int)
		 */
		default void setGrassColor(int color) {
			setGrassColor(Optional.of(color));
		}

		/**
		 * @see class_4763#getGrassColor()
		 * @see class_4763.class_4764#method_30822(int)
		 */
		default void setGrassColor(OptionalInt color) {
			color.ifPresentOrElse(this::setGrassColor, this::clearGrassColor);
		}

		/**
		 * @see class_4763#getGrassColor()
		 * @see class_4763.class_4764#method_30822(int)
		 */
		default void clearGrassColor() {
			setGrassColor(Optional.empty());
		}

		/**
		 * @see class_4763#getGrassColorModifier()
		 * @see class_4763.class_4764#method_30818(class_4763.class_5486)
		 */
		void setGrassColorModifier(class_4763.class_5486 colorModifier);

		/**
		 * @deprecated Set the music volume using environment attributes instead
		 * @see BiomeModificationContext#getAttributes()
		 * @see class_12206#field_63752
		 */
		@Deprecated
		void setMusicVolume(float volume);
	}

	interface GenerationSettingsContext {
		/**
		 * Removes a feature from one of this biomes generation steps, and returns if any features were removed.
		 */
		boolean removeFeature(class_2893.class_2895 step, class_5321<class_6796> placedFeatureKey);

		/**
		 * Removes a feature from all of this biomes generation steps, and returns if any features were removed.
		 */
		default boolean removeFeature(class_5321<class_6796> placedFeatureKey) {
			boolean anyFound = false;

			for (class_2893.class_2895 step : class_2893.class_2895.values()) {
				if (removeFeature(step, placedFeatureKey)) {
					anyFound = true;
				}
			}

			return anyFound;
		}

		/**
		 * Adds a feature to one of this biomes generation steps, identified by the placed feature's registry key.
		 */
		void addFeature(class_2893.class_2895 step, class_5321<class_6796> placedFeatureKey);

		/**
		 * Adds a configured carver to this biome.
		 */
		void addCarver(class_5321<class_2922<?>> carverKey);

		/**
		 * Removes all carvers with the given key from this biome.
		 *
		 * @return True if any carvers were removed.
		 */
		boolean removeCarver(class_5321<class_2922<?>> configuredCarverKey);
	}

	interface SpawnSettingsContext {
		/**
		 * Associated JSON property: <code>creature_spawn_probability</code>.
		 *
		 * @see class_5483#method_31002()
		 * @see class_5483.class_5496#method_31008(float)
		 */
		void setCreatureSpawnProbability(float probability);

		/**
		 * Provides a view of all spawns of the given spawn group.
		 *
		 * <p>Associated JSON property: <code>spawners</code>.
		 *
		 * @see class_5483#method_31004(class_1311)
		 */
		@UnmodifiableView List<class_6010<class_5483.class_1964>> getSpawnEntries(class_1311 spawnGroup);

		/**
		 * Associated JSON property: <code>spawners</code>.
		 *
		 * @see class_5483#method_31004(class_1311)
		 * @see class_5483.class_5496#method_31011(class_1311, int, class_5483.class_1964)
		 */
		void addSpawn(class_1311 spawnGroup, class_5483.class_1964 spawnEntry, int weight);

		/**
		 * Removes any spawns matching the given predicate from this biome, and returns true if any matched.
		 *
		 * <p>Associated JSON property: <code>spawners</code>.
		 */
		boolean removeSpawns(BiPredicate<class_1311, class_5483.class_1964> predicate);

		/**
		 * Removes all spawns of the given entity type.
		 *
		 * <p>Associated JSON property: <code>spawners</code>.
		 *
		 * @return True if any spawns were removed.
		 */
		default boolean removeSpawnsOfEntityType(class_1299<?> entityType) {
			return removeSpawns((spawnGroup, spawnEntry) -> spawnEntry.comp_3488() == entityType);
		}

		/**
		 * Removes all spawns of the given spawn group.
		 *
		 * <p>Associated JSON property: <code>spawners</code>.
		 */
		default void clearSpawns(class_1311 group) {
			removeSpawns((spawnGroup, spawnEntry) -> spawnGroup == group);
		}

		/**
		 * Removes all spawns.
		 *
		 * <p>Associated JSON property: <code>spawners</code>.
		 */
		default void clearSpawns() {
			removeSpawns((spawnGroup, spawnEntry) -> true);
		}

		/**
		 * Associated JSON property: <code>spawn_costs</code>.
		 *
		 * @see class_5483#method_31003(class_1299)
		 * @see class_5483.class_5496#method_31009(class_1299, double, double)
		 */
		void setSpawnCost(class_1299<?> entityType, double mass, double gravityLimit);

		/**
		 * Removes a spawn cost entry for a given entity type.
		 *
		 * <p>Associated JSON property: <code>spawn_costs</code>.
		 */
		void clearSpawnCost(class_1299<?> entityType);
	}
}
