/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.biome;

import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.stream.Collectors;
import net.minecraft.class_1959;
import net.minecraft.class_1972;
import net.minecraft.class_2169;
import net.minecraft.class_3756;
import net.minecraft.class_5321;
import net.minecraft.class_6544;
import net.minecraft.class_6880;
import net.minecraft.class_7871;
import com.google.common.base.Preconditions;
import it.unimi.dsi.fastutil.Hash;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenCustomHashMap;
import org.jspecify.annotations.Nullable;

/**
 * Internal data for modding Vanilla's {@link class_2169}.
 */
public final class TheEndBiomeData {
	public static final ThreadLocal<class_7871<class_1959>> biomeRegistry = new ThreadLocal<>();
	public static final Set<class_5321<class_1959>> ADDED_BIOMES = new HashSet<>();
	private static final Map<class_5321<class_1959>, WeightedPicker<class_5321<class_1959>>> END_BIOMES_MAP = new IdentityHashMap<>();
	private static final Map<class_5321<class_1959>, WeightedPicker<class_5321<class_1959>>> END_MIDLANDS_MAP = new IdentityHashMap<>();
	private static final Map<class_5321<class_1959>, WeightedPicker<class_5321<class_1959>>> END_BARRENS_MAP = new IdentityHashMap<>();

	static {
		END_BIOMES_MAP.computeIfAbsent(class_1972.field_9411, key -> new WeightedPicker<>())
				.add(class_1972.field_9411, 1.0);
		END_BIOMES_MAP.computeIfAbsent(class_1972.field_9442, key -> new WeightedPicker<>())
				.add(class_1972.field_9442, 1.0);
		END_BIOMES_MAP.computeIfAbsent(class_1972.field_9457, key -> new WeightedPicker<>())
				.add(class_1972.field_9457, 1.0);

		END_MIDLANDS_MAP.computeIfAbsent(class_1972.field_9442, key -> new WeightedPicker<>())
				.add(class_1972.field_9447, 1.0);
		END_BARRENS_MAP.computeIfAbsent(class_1972.field_9442, key -> new WeightedPicker<>())
				.add(class_1972.field_9465, 1.0);
	}

	private TheEndBiomeData() {
	}

	public static void addEndBiomeReplacement(class_5321<class_1959> replaced, class_5321<class_1959> variant, double weight) {
		Preconditions.checkNotNull(replaced, "replaced entry is null");
		Preconditions.checkNotNull(variant, "variant entry is null");
		Preconditions.checkArgument(weight > 0.0, "Weight is less than or equal to 0.0 (got %s)", weight);
		END_BIOMES_MAP.computeIfAbsent(replaced, key -> new WeightedPicker<>()).add(variant, weight);
		ADDED_BIOMES.add(variant);
	}

	public static void addEndMidlandsReplacement(class_5321<class_1959> highlands, class_5321<class_1959> midlands, double weight) {
		Preconditions.checkNotNull(highlands, "highlands entry is null");
		Preconditions.checkNotNull(midlands, "midlands entry is null");
		Preconditions.checkArgument(weight > 0.0, "Weight is less than or equal to 0.0 (got %s)", weight);
		END_MIDLANDS_MAP.computeIfAbsent(highlands, key -> new WeightedPicker<>()).add(midlands, weight);
		ADDED_BIOMES.add(midlands);
	}

	public static void addEndBarrensReplacement(class_5321<class_1959> highlands, class_5321<class_1959> barrens, double weight) {
		Preconditions.checkNotNull(highlands, "highlands entry is null");
		Preconditions.checkNotNull(barrens, "midlands entry is null");
		Preconditions.checkArgument(weight > 0.0, "Weight is less than or equal to 0.0 (got %s)", weight);
		END_BARRENS_MAP.computeIfAbsent(highlands, key -> new WeightedPicker<>()).add(barrens, weight);
		ADDED_BIOMES.add(barrens);
	}

	public static Overrides createOverrides(class_7871<class_1959> biomes) {
		return new Overrides(biomes);
	}

	/**
	 * An instance of this class is attached to each {@link class_2169}.
	 */
	public static class Overrides {
		public final Set<class_6880<class_1959>> customBiomes;

		// Vanilla entries to compare against
		private final class_6880<class_1959> endMidlands;
		private final class_6880<class_1959> endBarrens;
		private final class_6880<class_1959> endHighlands;

		// Maps where the keys have been resolved to actual entries
		private final @Nullable Map<class_6880<class_1959>, WeightedPicker<class_6880<class_1959>>> endBiomesMap;
		private final @Nullable Map<class_6880<class_1959>, WeightedPicker<class_6880<class_1959>>> endMidlandsMap;
		private final @Nullable Map<class_6880<class_1959>, WeightedPicker<class_6880<class_1959>>> endBarrensMap;

		// cache for our own sampler (used for random biome replacement selection)
		private final Map<class_6544.class_6552, class_3756> samplers = new WeakHashMap<>();

		public Overrides(class_7871<class_1959> biomeRegistry) {
			this.customBiomes = ADDED_BIOMES.stream().map(biomeRegistry::method_46747).collect(Collectors.toSet());

			this.endMidlands = biomeRegistry.method_46747(class_1972.field_9447);
			this.endBarrens = biomeRegistry.method_46747(class_1972.field_9465);
			this.endHighlands = biomeRegistry.method_46747(class_1972.field_9442);

			this.endBiomesMap = resolveOverrides(biomeRegistry, END_BIOMES_MAP, class_1972.field_9411);
			this.endMidlandsMap = resolveOverrides(biomeRegistry, END_MIDLANDS_MAP, class_1972.field_9447);
			this.endBarrensMap = resolveOverrides(biomeRegistry, END_BARRENS_MAP, class_1972.field_9465);
		}

		// Resolves all RegistryKey instances to RegistryEntries
		private @Nullable Map<class_6880<class_1959>, WeightedPicker<class_6880<class_1959>>> resolveOverrides(class_7871<class_1959> biomeRegistry, Map<class_5321<class_1959>, WeightedPicker<class_5321<class_1959>>> overrides, class_5321<class_1959> vanillaKey) {
			Map<class_6880<class_1959>, WeightedPicker<class_6880<class_1959>>> result = new Object2ObjectOpenCustomHashMap<>(overrides.size(), RegistryKeyHashStrategy.INSTANCE);

			for (Map.Entry<class_5321<class_1959>, WeightedPicker<class_5321<class_1959>>> entry : overrides.entrySet()) {
				WeightedPicker<class_5321<class_1959>> picker = entry.getValue();
				int count = picker.getEntryCount();
				if (count == 0 || (count == 1 && entry.getKey() == vanillaKey)) continue; // don't use no-op entries, for vanilla key biome check 1 as we have default entry

				result.put(biomeRegistry.method_46747(entry.getKey()), picker.map(biomeRegistry::method_46747));
			}

			return result.isEmpty() ? null : result;
		}

		public class_6880<class_1959> pick(int x, int y, int z, class_6544.class_6552 noise, class_6880<class_1959> vanillaBiome) {
			boolean isMidlands = vanillaBiome.method_40224(endMidlands::method_40225);

			if (isMidlands || vanillaBiome.method_40224(endBarrens::method_40225)) {
				// select a random highlands biome replacement, then try to replace it with a midlands or barrens biome replacement
				class_6880<class_1959> highlandsReplacement = pick(endHighlands, endHighlands, endBiomesMap, x, z, noise);
				Map<class_6880<class_1959>, WeightedPicker<class_6880<class_1959>>> map = isMidlands ? endMidlandsMap : endBarrensMap;

				return pick(highlandsReplacement, vanillaBiome, map, x, z, noise);
			} else {
				if (!END_BIOMES_MAP.containsKey(vanillaBiome.method_40230().orElseThrow())) {
					throw new IllegalStateException("Biome is not an End biome: " + vanillaBiome);
				}

				return pick(vanillaBiome, vanillaBiome, endBiomesMap, x, z, noise);
			}
		}

		private <T extends class_6880<class_1959>> T pick(T key, T defaultValue, Map<T, WeightedPicker<T>> pickers, int x, int z, class_6544.class_6552 noise) {
			if (pickers == null) return defaultValue;

			WeightedPicker<T> picker = pickers.get(key);
			if (picker == null) return defaultValue;
			int count = picker.getEntryCount();
			if (count == 0 || (count == 1 && key.method_40224(endHighlands::method_40225))) return defaultValue;

			// The x and z of the entry are divided by 64 to ensure custom biomes are large enough; going larger than this
			// seems to make custom biomes too hard to find.
			return picker.pickFromNoise(((MultiNoiseSamplerHooks) (Object) noise).fabric_getEndBiomesSampler(), x / 64.0, 0, z / 64.0);
		}
	}

	enum RegistryKeyHashStrategy implements Hash.Strategy<class_6880<?>> {
		INSTANCE;
		@Override
		public boolean equals(class_6880<?> a, class_6880<?> b) {
			if (a == b) return true;
			if (a == null || b == null) return false;
			if (a.method_40231() != b.method_40231()) return false;
			// This Optional#get is safe - if a has key, b should also have key
			// given a.getType() != b.getType() check above
			// noinspection OptionalGetWithoutIsPresent
			return a.method_40229().map(key -> b.method_40230().get() == key, b.comp_349()::equals);
		}

		@Override
		public int hashCode(class_6880<?> a) {
			if (a == null) return 0;
			return a.method_40229().map(System::identityHashCode, Object::hashCode);
		}
	}
}
