/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.client.datagen.v1.builder;

import java.util.Objects;

import com.mojang.serialization.Codec;
import org.jetbrains.annotations.ApiStatus;
import org.jspecify.annotations.Nullable;
import net.fabricmc.fabric.api.client.datagen.v1.provider.FabricSoundsProvider;
import net.fabricmc.fabric.impl.datagen.client.SoundTypeBuilderImpl;
import net.minecraft.class_156;
import net.minecraft.class_2960;
import net.minecraft.class_3414;
import net.minecraft.class_3419;
import net.minecraft.class_3542;
import net.minecraft.class_6880;

/**
 * Provides functionality for building entries that go into a {@code sounds.json} file.
 *
 * <p>Use in conjunction with {@link FabricSoundsProvider} to generate sound definitions.
 *
 * @see net.minecraft.class_1144
 * @see net.minecraft.class_1146
 */
@ApiStatus.NonExtendable
public interface SoundTypeBuilder {
	/**
	 * Creates a new builder pre-filled with a subtitle translation key string based on the passed event.
	 *
	 * <p>Note: To generate a translation value, use {@link net.fabricmc.fabric.api.datagen.v1.provider.FabricLanguageProvider.TranslationBuilder#add(class_3414, String)}.
	 *
	 * @return New sound type builder
	 */
	static SoundTypeBuilder of(class_3414 event) {
		Objects.requireNonNull(event, "Sound event cannot be null.");
		return of().subtitle(class_156.method_646("subtitles", event.comp_3319()));
	}

	/**
	 * Creates a new empty builder.
	 *
	 * @return New sound type builder
	 */
	static SoundTypeBuilder of() {
		return new SoundTypeBuilderImpl();
	}

	/**
	 * Sets the sound category the sound event must play on.
	 *
	 * <p>The default category is {@link class_3419#field_15254}. GUI elements should use {@link class_3419#field_15250}.
	 */
	SoundTypeBuilder category(class_3419 category);

	/**
	 * Sets an optional translation key string to use for the sound's subtitle.
	 *
	 * <p>The default is null (no subtitle).
	 *
	 * <p>Note: To generate a translation value, use {@link net.fabricmc.fabric.api.datagen.v1.provider.FabricLanguageProvider.TranslationBuilder#add(class_3414, String)}.
	 */
	SoundTypeBuilder subtitle(@Nullable String subtitle);

	/**
	 * Adds one sound to the event.
	 *
	 * @param sound base sound to add
	 */
	SoundTypeBuilder sound(EntryBuilder sound);

	/**
	 * Adds one or more sounds to the event.
	 *
	 * <p>This is a shorthand method for quickly adding multiple
	 * entries where each sound is a variant with an index at the end of their name.
	 *
	 * <p>Calling this with the count value of {@code 3} is the equivalent of doing:
	 *
	 * <p>{@code builder.sound(EntryBuilder.ofFile(id.withSuffixedPath("1"));}
	 * {@code builder.sound(EntryBuilder.ofFile(id.withSuffixedPath("2"));}
	 * {@code builder.sound(EntryBuilder.ofFile(id.withSuffixedPath("3"));}
	 *
	 * @param sound base sound to add
	 * @param count number of instances of that sound to register
	 */
	SoundTypeBuilder sound(EntryBuilder sound, int count);

	/**
	 * Represents the type of weighted sound entry.
	 *
	 * @see net.minecraft.class_1111.class_1112
	 */
	enum RegistrationType implements class_3542 {
		/**
		 * Direct references to sound files by path and filename excluding {@code *.ogg} extension.
		 */
		FILE("file"),
		/**
		 * References to another sound event.
		 */
		SOUND_EVENT("event");

		public static final Codec<RegistrationType> CODEC = class_3542.method_28140(RegistrationType::values);

		private final String name;

		RegistrationType(String name) {
			this.name = name;
		}

		@Override
		public String method_15434() {
			return name;
		}
	}

	/**
	 * Builder for creating a weighted sound entry that can be played for a particular sound type.
	 */
	@ApiStatus.NonExtendable
	interface EntryBuilder {
		/**
		 * The default sound volume.
		 */
		float DEFAULT_VOLUME = 1F;

		/**
		 * The default sound pitch.
		 */
		float DEFAULT_PITCH = 1F;

		/**
		 * The default weight applied to individual sounds.
		 */
		int DEFAULT_WEIGHT = 1;

		/**
		 * The default attenuation distance for a sound (16 blocks).
		 */
		int DEFAULT_ATTENUATION_DISTANCE = 16;

		/**
		 * Creates a builder for constructing a new sound entry.
		 *
		 * @param id sound file or event
		 */
		static EntryBuilder create(RegistrationType type, class_2960 id) {
			return SoundTypeBuilderImpl.EntryBuilderImpl.create(type, id);
		}

		/**
		 * Creates a builder for constructing a new sound entry.
		 *
		 * @param soundFile sound file excluding the {@code .ogg} extension
		 */
		static EntryBuilder ofFile(class_2960 soundFile) {
			return SoundTypeBuilderImpl.EntryBuilderImpl.ofFile(soundFile);
		}

		/**
		 * Creates a builder for constructing a new sound entry.
		 *
		 * @param event the sound event
		 */
		static EntryBuilder ofEvent(class_3414 event) {
			return SoundTypeBuilderImpl.EntryBuilderImpl.ofEvent(event);
		}

		/**
		 * Creates a builder for constructing a new sound entry.
		 *
		 * @param event the sound event
		 */
		static EntryBuilder ofEvent(class_6880<class_3414> event) {
			return SoundTypeBuilderImpl.EntryBuilderImpl.ofEvent(event);
		}

		/**
		 * Sets the volume of the sound.
		 *
		 * <p>Must be a value between {@code 0} and {@code 1} (inclusive).
		 *
		 * <p>The default volume is {@value EntryBuilder#DEFAULT_VOLUME}.
		 *
		 * @see net.minecraft.class_1140#VOLUME_MIN
		 * @see net.minecraft.class_1140#VOLUME_MAX
		 */
		EntryBuilder volume(float volume);

		/**
		 * Sets the pitch of the sound.
		 *
		 * <p>Must be a value between {@code 0.5} and {@code 2}.
		 *
		 * <p>The default pitch is {@link EntryBuilder#DEFAULT_PITCH} ({@code 1F}).
		 *
		 * @see net.minecraft.class_1140#PITCH_MIN
		 * @see net.minecraft.class_1140#PITCH_MAX
		 */
		EntryBuilder pitch(float pitch);

		/**
		 * Sets the attenuation block distance of the sound.
		 *
		 * <p>The default attenuation is {@value EntryBuilder#DEFAULT_ATTENUATION_DISTANCE} blocks. Setting it to
		 * higher will cause the sound to be heard from greater distances.
		 */
		EntryBuilder attenuationDistance(int attenuationDistance);

		/**
		 * Sets the weight or "chance" that this sound has of playing when
		 * its parent sound event is called upon.
		 *
		 * <p>The default weight is {@value EntryBuilder#DEFAULT_WEIGHT}.
		 */
		EntryBuilder weight(int weight);

		/**
		 * Configures the sound to be streamed.
		 * This is usually set for longer sounds like music discs
		 * to prevent delays when the game tries to play them.
		 *
		 * <p>The default value is {@code false}.
		 */
		EntryBuilder stream(boolean stream);

		/**
		 * Configures whether the sound must be preloaded by the game.
		 * By default, sounds are only loaded upon playing.
		 *
		 * <p>Setting this to {@code true} will cause them to be loaded when the game starts.
		 *
		 * <p>The default value is {@code false}.
		 */
		EntryBuilder preload(boolean preload);
	}
}
