/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.entity;

import java.util.Map;
import java.util.Objects;
import java.util.OptionalInt;
import java.util.UUID;

import com.google.common.collect.MapMaker;
import com.mojang.authlib.GameProfile;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.impl.event.interaction.FakePlayerNetworkHandler;
import net.minecraft.class_1263;
import net.minecraft.class_1282;
import net.minecraft.class_1297;
import net.minecraft.class_1496;
import net.minecraft.class_2338;
import net.minecraft.class_2625;
import net.minecraft.class_268;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3445;
import net.minecraft.class_3908;
import net.minecraft.class_8791;

/**
 * A "fake player" is a {@link class_3222} that is not a human player.
 * They are typically used to automatically perform player actions such as placing blocks.
 *
 * <p>The easiest way to obtain a fake player is with {@link FakePlayer#get(class_3218)} or {@link FakePlayer#get(class_3218, GameProfile)}.
 * It is also possible to create a subclass for more control over the fake player's behavior.
 *
 * <p>For good inter-mod compatibility, fake players should have the UUID of their owning (human) player.
 * They should still have a different name to ensure the {@link GameProfile} is different.
 * For example:
 * <pre>{@code
 * UUID humanPlayerUuid = ...;
 * String humanPlayerName = ...;
 * GameProfile fakeProfile = new GameProfile(humanPlayerUuid, "[Block Breaker of " + humanPlayerName + "]");
 * }</pre>
 * If a fake player does not belong to a specific player, the {@link #DEFAULT_UUID default UUID} should be used.
 *
 * <p>Fake players try to behave like regular {@link class_3222} objects to a reasonable extent.
 * In some edge cases, or for gameplay considerations, it might be necessary to check whether a {@link class_3222} is a fake player.
 * This can be done with an {@code instanceof} check: {@code player instanceof FakePlayer}.
 */
public class FakePlayer extends class_3222 {
	/**
	 * Default UUID, for fake players not associated with a specific (human) player.
	 */
	public static final UUID DEFAULT_UUID = UUID.fromString("41C82C87-7AfB-4024-BA57-13D2C99CAE77");
	private static final GameProfile DEFAULT_PROFILE = new GameProfile(DEFAULT_UUID, "[Minecraft]");

	/**
	 * Retrieves a fake player for the specified world, using the {@link #DEFAULT_UUID default UUID}.
	 * This is suitable when the fake player is not associated with a specific (human) player.
	 * Otherwise, the UUID of the owning (human) player should be used (see class javadoc).
	 *
	 * <p>Instances are reused for the same world parameter.
	 *
	 * <p>Caution should be exerted when storing the returned value,
	 * as strong references to the fake player will keep the world loaded.
	 */
	public static FakePlayer get(class_3218 world) {
		return get(world, DEFAULT_PROFILE);
	}

	/**
	 * Retrieves a fake player for the specified world and game profile.
	 * See class javadoc for more information on fake player game profiles.
	 *
	 * <p>Instances are reused for the same parameters.
	 *
	 * <p>Caution should be exerted when storing the returned value,
	 * as strong references to the fake player will keep the world loaded.
	 */
	public static FakePlayer get(class_3218 world, GameProfile profile) {
		Objects.requireNonNull(world, "World may not be null.");
		Objects.requireNonNull(profile, "Game profile may not be null.");

		return FAKE_PLAYER_MAP.computeIfAbsent(new FakePlayerKey(world, profile), key -> new FakePlayer(key.world, key.profile));
	}

	private record FakePlayerKey(class_3218 world, GameProfile profile) { }
	private static final Map<FakePlayerKey, FakePlayer> FAKE_PLAYER_MAP = new MapMaker().weakValues().makeMap();

	protected FakePlayer(class_3218 world, GameProfile profile) {
		super(world.method_8503(), world, profile, class_8791.method_53821());

		this.field_13987 = new FakePlayerNetworkHandler(this);
	}

	@Override
	public void method_5773() { }

	@Override
	public void method_14213(class_8791 settings) { }

	@Override
	public void method_7342(class_3445<?> stat, int amount) { }

	@Override
	public void method_7266(class_3445<?> stat) { }

	@Override
	public boolean method_5679(class_3218 world, class_1282 damageSource) {
		return true;
	}

	@Nullable
	@Override
	public class_268 method_5781() {
		// Scoreboard team is checked using the gameprofile name by default, which we don't want.
		return null;
	}

	@Override
	public void method_18403(class_2338 pos) {
		// Don't lock bed forever.
	}

	@Override
	public boolean method_5873(class_1297 entity, boolean force, boolean emitEvent) {
		return false;
	}

	@Override
	public void method_7311(class_2625 sign, boolean front) { }

	@Override
	public OptionalInt method_17355(@Nullable class_3908 factory) {
		return OptionalInt.empty();
	}

	@Override
	public void method_7291(class_1496 horse, class_1263 inventory) { }
}
