/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.registry.sync;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;

import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import it.unimi.dsi.fastutil.objects.Object2IntLinkedOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import org.jspecify.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.minecraft.class_11560;
import net.minecraft.class_2378;
import net.minecraft.class_2561;
import net.minecraft.class_2596;
import net.minecraft.class_2960;
import net.minecraft.class_7923;
import net.minecraft.class_8605;
import net.minecraft.class_8610;
import net.minecraft.server.MinecraftServer;
import net.fabricmc.fabric.api.event.registry.RegistryAttribute;
import net.fabricmc.fabric.api.event.registry.RegistryAttributeHolder;
import net.fabricmc.fabric.api.networking.v1.ServerConfigurationNetworking;
import net.fabricmc.fabric.impl.networking.server.ServerNetworkingImpl;
import net.fabricmc.fabric.impl.registry.sync.packet.DirectRegistryPacketHandler;

public final class RegistrySyncManager {
	public static final boolean DEBUG = Boolean.getBoolean("fabric.registry.debug");

	public static final DirectRegistryPacketHandler DIRECT_PACKET_HANDLER = new DirectRegistryPacketHandler();

	private static final Logger LOGGER = LoggerFactory.getLogger("FabricRegistrySync");
	private static final boolean DEBUG_WRITE_REGISTRY_DATA = Boolean.getBoolean("fabric.registry.debug.writeContentsAsCsv");

	//Set to true after vanilla's bootstrap has completed
	public static boolean postBootstrap = false;

	private RegistrySyncManager() { }

	public static void configureClient(class_8610 handler, MinecraftServer server) {
		if (!DEBUG && server.method_19466(new class_11560(handler.method_52404()))) {
			// Dont send in singleplayer
			return;
		}

		final Map<class_2960, Object2IntMap<class_2960>> map = RegistrySyncManager.createAndPopulateRegistryMap();

		if (map == null) {
			// Don't send when there is nothing to map
			return;
		}

		if (!ServerConfigurationNetworking.canSend(handler, DIRECT_PACKET_HANDLER.getPacketId())) {
			if (areAllRegistriesOptional(map)) {
				// Allow the client to connect if all of the registries we want to sync are optional
				return;
			}

			// Disconnect incompatible clients
			class_2561 message = getIncompatibleClientText(ServerNetworkingImpl.getAddon(handler).getClientBrand(), map);
			handler.method_52396(message);
			return;
		}

		handler.addTask(new SyncConfigurationTask(handler, map));
	}

	private static class_2561 getIncompatibleClientText(@Nullable String brand, Map<class_2960, Object2IntMap<class_2960>> map) {
		String brandText = switch (brand) {
		case "fabric" -> "Fabric API";
		case null, default -> "Fabric Loader and Fabric API";
		};

		final int toDisplay = 4;

		List<String> namespaces = map.values().stream()
				.map(Object2IntMap::keySet)
				.flatMap(Set::stream)
				.map(Identifier::getNamespace)
				.filter(s -> !s.equals(Identifier.DEFAULT_NAMESPACE))
				.distinct()
				.sorted()
				.toList();

		MutableText text = Text.literal("The following registry entry namespaces may be related:\n\n");

		for (int i = 0; i < Math.min(namespaces.size(), toDisplay); i++) {
			text = text.append(Text.literal(namespaces.get(i)).formatted(Formatting.YELLOW));
			text = text.append(ScreenTexts.LINE_BREAK);
		}

		if (namespaces.size() > toDisplay) {
			text = text.append(Text.literal("And %d more...".formatted(namespaces.size() - toDisplay)));
		}

		return Text.literal("This server requires ").append(Text.literal(brandText).formatted(Formatting.GREEN)).append(" installed on your client!")
				.append(ScreenTexts.LINE_BREAK).append(text)
				.append(ScreenTexts.LINE_BREAK).append(ScreenTexts.LINE_BREAK).append(Text.literal("Contact the server's administrator for more information!").formatted(Formatting.GOLD));
	}

	private static boolean areAllRegistriesOptional(Map<class_2960, Object2IntMap<class_2960>> map) {
		return map.keySet().stream()
				.map(class_7923.field_41167::method_63535)
				.filter(Objects::nonNull)
				.map(RegistryAttributeHolder::get)
				.allMatch(attributes -> attributes.hasAttribute(RegistryAttribute.OPTIONAL));
	}

	public record SyncConfigurationTask(
			class_8610 handler,
			Map<class_2960, Object2IntMap<class_2960>> map
	) implements class_8605 {
		public static final class_8606 KEY = new class_8606("fabric:registry/sync");

		@Override
		public void method_52376(Consumer<class_2596<?>> sender) {
			DIRECT_PACKET_HANDLER.sendPacket(payload -> handler.method_14364(ServerConfigurationNetworking.createS2CPacket(payload)), map);
		}

		@Override
		public class_8606 method_52375() {
			return KEY;
		}
	}

	/**
	 * Creates a {@link Map} used to sync the registry ids.
	 *
	 * @return a {@link Map} to sync, null when empty
	 */
	@Nullable
	public static Map<class_2960, Object2IntMap<class_2960>> createAndPopulateRegistryMap() {
		Map<class_2960, Object2IntMap<class_2960>> map = new LinkedHashMap<>();

		for (class_2960 registryId : class_7923.field_41167.method_10235()) {
			class_2378 registry = class_7923.field_41167.method_63535(registryId);

			if (DEBUG_WRITE_REGISTRY_DATA) {
				File location = new File(".fabric" + File.separatorChar + "debug" + File.separatorChar + "registry");
				boolean c = true;

				if (!location.exists()) {
					if (!location.mkdirs()) {
						LOGGER.warn("[fabric-registry-sync debug] Could not create " + location.getAbsolutePath() + " directory!");
						c = false;
					}
				}

				if (c && registry != null) {
					File file = new File(location, registryId.toString().replace(':', '.').replace('/', '.') + ".csv");

					try (FileOutputStream stream = new FileOutputStream(file)) {
						StringBuilder builder = new StringBuilder("Raw ID,String ID,Class Type\n");

						for (Object o : registry) {
							String classType = (o == null) ? "null" : o.getClass().getName();
							//noinspection unchecked
							class_2960 id = registry.method_10221(o);
							if (id == null) continue;

							//noinspection unchecked
							int rawId = registry.method_10206(o);
							String stringId = id.toString();
							builder.append("\"").append(rawId).append("\",\"").append(stringId).append("\",\"").append(classType).append("\"\n");
						}

						stream.write(builder.toString().getBytes(StandardCharsets.UTF_8));
					} catch (IOException e) {
						LOGGER.warn("[fabric-registry-sync debug] Could not write to " + file.getAbsolutePath() + "!", e);
					}
				}
			}

			RegistryAttributeHolder attributeHolder = RegistryAttributeHolder.get(registry.method_46765());

			if (!attributeHolder.hasAttribute(RegistryAttribute.SYNCED)) {
				LOGGER.debug("Not syncing registry: {}", registryId);
				continue;
			}

			/*
			 * Dont do anything with vanilla registries on client sync.
			 *
			 * This will not sync IDs if a world has been previously modded, either from removed mods
			 * or a previous version of fabric registry sync.
			 */
			if (!attributeHolder.hasAttribute(RegistryAttribute.MODDED)) {
				LOGGER.debug("Skipping un-modded registry: " + registryId);
				continue;
			}

			LOGGER.debug("Syncing registry: " + registryId);

			if (registry instanceof RemappableRegistry) {
				Object2IntMap<class_2960> idMap = new Object2IntLinkedOpenHashMap<>();
				IntSet rawIdsFound = DEBUG ? new IntOpenHashSet() : null;

				for (Object o : registry) {
					//noinspection unchecked
					class_2960 id = registry.method_10221(o);
					if (id == null) continue;

					//noinspection unchecked
					int rawId = registry.method_10206(o);

					if (DEBUG) {
						if (registry.method_63535(id) != o) {
							LOGGER.error("[fabric-registry-sync] Inconsistency detected in " + registryId + ": object " + o + " -> string ID " + id + " -> object " + registry.method_63535(id) + "!");
						}

						if (registry.method_10200(rawId) != o) {
							LOGGER.error("[fabric-registry-sync] Inconsistency detected in " + registryId + ": object " + o + " -> integer ID " + rawId + " -> object " + registry.method_10200(rawId) + "!");
						}

						if (!rawIdsFound.add(rawId)) {
							LOGGER.error("[fabric-registry-sync] Inconsistency detected in " + registryId + ": multiple objects hold the raw ID " + rawId + " (this one is " + id + ")");
						}
					}

					idMap.put(id, rawId);
				}

				map.put(registryId, idMap);
			}
		}

		if (map.isEmpty()) {
			return null;
		}

		return map;
	}

	public static void bootstrapRegistries() {
		postBootstrap = true;
	}
}
