/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.transfer.v1.fluid;

import java.util.Optional;
import net.minecraft.class_156;
import net.minecraft.class_1755;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2561;
import net.minecraft.class_3414;
import net.minecraft.class_3609;
import net.minecraft.class_3611;
import net.minecraft.class_7923;
import org.jspecify.annotations.Nullable;

/**
 * Defines the common attributes of {@linkplain FluidVariant fluid variants} of a given Fluid.
 * Register with {@link FluidVariantAttributes#register}.
 */
public interface FluidVariantAttributeHandler {
	/**
	 * Return the name that should be used for the passed fluid variant.
	 */
	default class_2561 getName(FluidVariant fluidVariant) {
		class_2248 fluidBlock = fluidVariant.getFluid().method_15785().method_15759().method_26204();

		if (!fluidVariant.isBlank() && fluidBlock == class_2246.field_10124) {
			// Some non-placeable fluids use air as their fluid block, in that case infer translation key from the fluid id.
			return class_2561.method_43471(class_156.method_646("block", class_7923.field_41173.method_10221(fluidVariant.getFluid())));
		} else {
			return fluidBlock.method_9518();
		}
	}

	/**
	 * Return the sound corresponding to this fluid being filled, or none if no sound is available.
	 *
	 * <p>If a non-empty sound event is returned, {@link class_3611#method_32359} will return that sound.
	 */
	default Optional<class_3414> getFillSound(FluidVariant variant) {
		return Optional.empty();
	}

	/**
	 * Return the sound corresponding to this fluid being emptied, or none if no sound is available.
	 *
	 * <p>If a non-empty sound event is returned, {@link class_1755#method_7727} will play that sound.
	 */
	default Optional<class_3414> getEmptySound(FluidVariant variant) {
		return Optional.empty();
	}

	/**
	 * Return an integer in [0, 15]: the light level emitted by this fluid, or 0 if it doesn't naturally emit light.
	 */
	default int getLuminance(FluidVariant variant) {
		return variant.getFluid().method_15785().method_15759().method_26213();
	}

	/**
	 * Return a non-negative integer, representing the temperature of this fluid in Kelvin.
	 * The reference values are {@value FluidConstants#WATER_TEMPERATURE} for water, and {@value FluidConstants#LAVA_TEMPERATURE} for lava.
	 */
	default int getTemperature(FluidVariant variant) {
		return FluidConstants.WATER_TEMPERATURE;
	}

	/**
	 * Return a positive integer, representing the viscosity of this fluid.
	 * Fluids with lower viscosity generally flow faster than fluids with higher viscosity.
	 *
	 * <p>More precisely, viscosity should be {@value FluidConstants#VISCOSITY_RATIO} * {@link class_3609#method_15733} for flowable fluids.
	 * The reference values are {@value FluidConstants#WATER_VISCOSITY} for water,
	 * {@value FluidConstants#LAVA_VISCOSITY_NETHER} for lava in ultrawarm dimensions (such as the nether),
	 * and {@value FluidConstants#LAVA_VISCOSITY} for lava in other dimensions.
	 *
	 * @param world World if available, otherwise null.
	 */
	default int getViscosity(FluidVariant variant, @Nullable class_1937 world) {
		return FluidConstants.WATER_VISCOSITY;
	}

	/**
	 * Return true if this fluid is lighter than air.
	 * Fluids that are lighter than air generally flow upwards.
	 */
	default boolean isLighterThanAir(FluidVariant variant) {
		return false;
	}
}
