/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

package org.cadixdev.mercury.mixin;

import static org.cadixdev.mercury.mixin.annotation.AccessorType.FIELD_GETTER;
import static org.cadixdev.mercury.mixin.util.MixinConstants.*;
import static org.cadixdev.mercury.util.BombeBindings.convertType;

import org.cadixdev.bombe.analysis.InheritanceProvider;
import org.cadixdev.bombe.type.FieldType;
import org.cadixdev.bombe.type.MethodDescriptor;
import org.cadixdev.bombe.type.Type;
import org.cadixdev.bombe.type.VoidType;
import org.cadixdev.bombe.type.signature.FieldSignature;
import org.cadixdev.bombe.type.signature.MethodSignature;
import org.cadixdev.lorenz.MappingSet;
import org.cadixdev.lorenz.model.*;
import org.cadixdev.mercury.RewriteContext;
import org.cadixdev.mercury.analysis.MercuryInheritanceProvider;
import org.cadixdev.mercury.mixin.annotation.AccessorData;
import org.cadixdev.mercury.mixin.annotation.AccessorName;
import org.cadixdev.mercury.mixin.annotation.AccessorType;
import org.cadixdev.mercury.mixin.annotation.AtData;
import org.cadixdev.mercury.mixin.annotation.DescData;
import org.cadixdev.mercury.mixin.annotation.InjectData;
import org.cadixdev.mercury.mixin.annotation.InjectTarget;
import org.cadixdev.mercury.mixin.annotation.MixinClass;
import org.cadixdev.mercury.mixin.annotation.ShadowData;
import org.cadixdev.mercury.mixin.annotation.SliceData;
import org.cadixdev.mercury.util.BombeBindings;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.AST;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.ASTVisitor;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.Annotation;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.ArrayInitializer;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.Expression;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.IAnnotationBinding;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.IBinding;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.IExtendedModifier;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.IMemberValuePairBinding;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.IMethodBinding;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.ITypeBinding;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.IVariableBinding;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.InfixExpression;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.MemberValuePair;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.MethodDeclaration;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.NormalAnnotation;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.SimpleName;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.SingleMemberAnnotation;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.StringLiteral;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.TypeDeclaration;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

public class MixinRemapperVisitor extends ASTVisitor {

    final RewriteContext context;
    final MappingSet mappings;
    private final InheritanceProvider inheritanceProvider;

    MixinRemapperVisitor(final RewriteContext context, final MappingSet mappings) {
        this.context = context;
        this.mappings = mappings;
        this.inheritanceProvider = MercuryInheritanceProvider.get(context.getMercury());
    }

    private void remapPrivateMixinTarget(final AST ast, final TypeDeclaration typeDeclaration, final ITypeBinding binding) {
        for (final Object rawModifier : typeDeclaration.modifiers()) {
            final IExtendedModifier modifier = (IExtendedModifier) rawModifier;
            if (!modifier.isAnnotation()) return;
            final Annotation rawAnnot = (Annotation) modifier;

            if (rawAnnot.isNormalAnnotation()) {
                final NormalAnnotation annot = (NormalAnnotation) rawAnnot;

                for (final Object raw : annot.values()) {
                    final MemberValuePair pair = (MemberValuePair) raw;

                    if (Objects.equals("targets", pair.getName().getIdentifier())) {
                        final Expression targets = pair.getValue();

                        if (targets instanceof StringLiteral) {
                            final StringLiteral target = (StringLiteral) targets;
                            this.remapPrivateMixinTargetLiteral(ast, target);
                        }
                        else if (targets instanceof ArrayInitializer) {
                            final ArrayInitializer target = (ArrayInitializer) targets;

                            for (final Object expression : target.expressions()) {
                                if (expression instanceof StringLiteral) {
                                    this.remapPrivateMixinTargetLiteral(ast, (StringLiteral) expression);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    private void remapPrivateMixinTargetLiteral(final AST ast, final StringLiteral literal) {
        final String className = literal.getLiteralValue();
        if (className.isEmpty()) return;
        final boolean binaryFormat = className.contains("/");

        ClassMapping<?, ?> classMapping = this.mappings.getTopLevelClassMapping(className).orElse(null);
        if (classMapping == null) {
            classMapping = this.mappings.getClassMapping(className).orElse(null);
        }

        if (classMapping != null) {
            final String remappedClassName = classMapping.getFullDeobfuscatedName();
            replaceExpression(ast, this.context, literal, binaryFormat ?
                    remappedClassName :
                    remappedClassName.replace('/', '.'));
        }
    }

    void remapField(final SimpleName node, final IVariableBinding binding) {
        if (!binding.isField()) return;

        final ITypeBinding declaringClass = binding.getDeclaringClass();
        if (declaringClass == null) return;

        final MixinClass mixin = MixinClass.fetch(declaringClass, this.mappings);
        if (mixin == null) return;

        // todo: support multiple targets properly
        final ClassMapping<?, ?> target = this.mappings.getOrCreateClassMapping(mixin.getTargetNames()[0]);

        for (final IAnnotationBinding annotation : binding.getAnnotations()) {
            final String annotationType = annotation.getAnnotationType().getBinaryName();

            // @Shadow
            if (Objects.equals(SHADOW_CLASS, annotationType)) {
                final ShadowData shadow = ShadowData.from(annotation);

                final boolean usedPrefix = binding.getName().startsWith(shadow.getPrefix());
                final FieldSignature targetSignature = convertSignature(shadow.stripPrefix(binding.getName()), binding.getType());
                final FieldSignature mixinSignature = BombeBindings.convertSignature(binding);

                // Copy de-obfuscation mapping
                mixin.copyFieldMapping(
                        target,
                        mixinSignature,
                        targetSignature,
                        deobfName -> usedPrefix ? shadow.prefix(deobfName) : deobfName
                );
            }
        }
    }

    @Override
    public boolean visit(final MethodDeclaration node) {
        final AST ast = this.context.getCompilationUnit().getAST();
        final IMethodBinding binding = node.resolveBinding();

        if (binding == null) {
            if (this.context.getMercury().isGracefulClasspathChecks()) {
                return true;
            }

            throw new IllegalStateException("No binding for type declaration " + node.getName() + " in class " + this.context.getQualifiedPrimaryType());
        }

        final ITypeBinding declaringClass = binding.getDeclaringClass();
        final MixinClass mixin = MixinClass.fetch(declaringClass, this.mappings);
        if (mixin == null) return true;

        // @Implements
        if (node.getName().getIdentifier().contains("$")) {
            final String[] split = node.getName().getIdentifier().split("\\$");
            final String prefix = split[0];
            final String name = split[1];

            // check we implement something
            if (mixin.getImplementsData().containsKey(prefix)) {
                final ITypeBinding iface = mixin.getImplementsData().get(prefix);
                final ClassMapping<?, ?> target = this.mappings.getOrCreateClassMapping(iface.getBinaryName());

                final MethodSignature targetSignature = convertSignature(name, binding);
                final MethodSignature mixinSignature = BombeBindings.convertSignature(binding);

                // Copy de-obfuscation mapping
                mixin.copyMethodMapping(
                        target,
                        mixinSignature,
                        targetSignature,
                        deobfName -> prefix + "$" + deobfName
                );

                return true;
            }
        }

        // todo: support multiple targets properly
        final ClassMapping<?, ?> target = this.mappings.getOrCreateClassMapping(mixin.getTargetNames()[0]);

        // todo: only complete the mixin we are targeting
        for (final ITypeBinding mixinTarget : mixin.getTargets(this.context.getMercury())) {
            target.complete(this.inheritanceProvider, mixinTarget);
        }

        for (int i = 0; i < binding.getAnnotations().length; i++) {
            final IAnnotationBinding annotation = binding.getAnnotations()[i];
            final String annotationType = annotation.getAnnotationType().getBinaryName();

            // @Shadow
            if (Objects.equals(SHADOW_CLASS, annotationType)) {
                final ShadowData shadow = ShadowData.from(annotation);

                final boolean usedPrefix = binding.getName().startsWith(shadow.getPrefix());
                final MethodSignature targetSignature = convertSignature(shadow.stripPrefix(binding.getName()), binding);
                final MethodSignature mixinSignature = BombeBindings.convertSignature(binding);

                // Copy de-obfuscation mapping
                mixin.copyMethodMapping(
                        target,
                        mixinSignature,
                        targetSignature,
                        deobfName -> usedPrefix ? shadow.prefix(deobfName) : deobfName
                );
            }

            // @Overwrite
            if (Objects.equals(OVERWRITE_CLASS, annotationType)) {
                final MethodSignature signature = BombeBindings.convertSignature(binding);

                // Copy de-obfuscation mapping
                mixin.copyMethodMapping(target, signature, s -> s);
            }

            // @Accessor and @Invoker
            if (Objects.equals(ACCESSOR_CLASS, annotationType) || Objects.equals(INVOKER_CLASS, annotationType)) {
                final AccessorName name = AccessorName.of(binding.getName());
                final AccessorData accessor = AccessorData.from(annotation);
                final MethodSignature mixinSignature = BombeBindings.convertSignature(binding);
                final AccessorType type = AccessorType.get(
                        Objects.equals(INVOKER_CLASS, annotationType),
                        binding, mixinSignature, accessor
                );

                // Inflect target from target name, if not set in annotation
                final boolean inflect = accessor.getTarget().isEmpty();
                final String targetName = inflect ? name.getName() : accessor.getTarget();

                switch (type) {
                    // @Accessor
                    case FIELD_GETTER:
                    case FIELD_SETTER: {
                        final FieldSignature targetSignature = new FieldSignature(targetName, type == FIELD_GETTER ?
                                // For getters, use the return type
                                (FieldType) mixinSignature.getDescriptor().getReturnType() :
                                // For setters, use the first argument in the method
                                mixinSignature.getDescriptor().getParamTypes().get(0)
                        );

                        // Get mapping of target field
                        final FieldMapping targetField = target.computeFieldMapping(targetSignature).orElse(null);
                        if (targetField == null) continue;

                        // Inflect target name from name of method
                        if (inflect) {
                            mixin.copyMethodMapping(target, mixinSignature, targetSignature, name::prefix);
                        }
                        else {
                            final Annotation rawAnnotation = (Annotation) node.modifiers().get(i);
                            replaceValueInAnnotation(ast, this.context, rawAnnotation, targetField.getDeobfuscatedName());
                        }
                        break;
                    }

                    // @Invoker
                    case METHOD_PROXY: {
                        final MethodSignature targetSignature = new MethodSignature(targetName, mixinSignature.getDescriptor());

                        // Get mapping of target field
                        final MethodMapping targetMethod = target.getMethodMapping(targetSignature).orElse(null);
                        if (targetMethod == null) continue;

                        // Inflect target name from name of method
                        if (inflect) {
                            mixin.copyMethodMapping(target, mixinSignature, targetSignature, name::prefix);
                        }
                        else {
                            final Annotation rawAnnotation = (Annotation) node.modifiers().get(i);
                            replaceValueInAnnotation(ast, this.context, rawAnnotation, targetMethod.getDeobfuscatedName());
                        }
                        break;
                    }
                    case OBJECT_FACTORY: {
                        // @Invoker.value will always be either <init> or the target class name
                        if (!Objects.equals("<init>", accessor.getTarget())) {
                            // Remap target class name
                            final ClassMapping<?, ?> targetClass = this.mappings.computeClassMapping(accessor.getTarget()).orElse(null);
                            if (targetClass == null) continue;

                            final Annotation rawAnnotation = (Annotation) node.modifiers().get(i);
                            replaceValueInAnnotation(ast, this.context, rawAnnotation, targetClass.getFullDeobfuscatedName());
                        }
                        break;
                    }
                }
            }

            // @Inject, @Redirect, @ModifyConstant, & @ModifyVariable
            if (Objects.equals(INJECT_CLASS, annotationType)
                    || Objects.equals(REDIRECT_CLASS, annotationType)
                    || Objects.equals(MODIFY_CONSTANT_CLASS, annotationType)
                    || Objects.equals(MODIFY_VARIABLE_CLASS, annotationType)
                    || Objects.equals(MODIFY_ARG_CLASS, annotationType)
                    || Objects.equals(WRAP_METHOD, annotationType)
                    || Objects.equals(WRAP_OPERATION_VALUE, annotationType)
                    || Objects.equals(WRAP_WITH_CONDITION, annotationType)
                    || Objects.equals(WRAP_WITH_CONDITION_V2, annotationType)
                    || Objects.equals(MODIFY_EXPRESSION_VALUE, annotationType)
                    || Objects.equals(MODIFY_RECEIVER, annotationType)
                    || Objects.equals(MODIFY_RETURN_VALUE, annotationType)) {
                final InjectData inject = InjectData.from(annotation);

                // Find target method(s?)
                // todo: implement selectors
                final String[] injectTargets = new String[inject.getInjectTargets().length];
                for (int j = 0; j < inject.getInjectTargets().length; j++) {
                    final InjectTarget injectTarget = inject.getInjectTargets()[j];
                    injectTargets[j] = remapInjectTarget(target, injectTarget, binding);
                }

                final NormalAnnotation originalAnnotation = (NormalAnnotation) node.modifiers().get(i);
                int atIndex = 0;
                int sliceIndex = 0;
                for (final Object raw : originalAnnotation.values()) {
                    final MemberValuePair pair = (MemberValuePair) raw;

                    // Remap the method pair
                    // TODO: handle the case where we point towards a string constant?
                    if (Objects.equals("method", pair.getName().getIdentifier())) {
                        remapMethod(ast, pair, injectTargets);
                    }

                    // Remap the @Desc targets
                    if ("target".equals(pair.getName().getIdentifier())) {
                        if (pair.getValue() instanceof Annotation) {
                            final Annotation original = (Annotation) pair.getValue();
                            this.remapDescAnnotation(ast, declaringClass, original, inject.getDescTargets()[0]);
                        }
                        else if (pair.getValue() instanceof ArrayInitializer) {
                            final ArrayInitializer array = (ArrayInitializer) pair.getValue();
                            for (int j = 0; j < array.expressions().size(); j++) {
                                final Annotation original = (Annotation) array.expressions().get(j);
                                this.remapDescAnnotation(ast, declaringClass, original, inject.getDescTargets()[j]);
                            }
                        }
                    }

                    // Remap @At
                    if (Objects.equals("at", pair.getName().getIdentifier())) {
                        // it could be a SingleMemberAnnotation here but we don't care about that case

                        if (pair.getValue() instanceof ArrayInitializer) {
                            final ArrayInitializer value = (ArrayInitializer) pair.getValue();

                            for (final Object expression : value.expressions()) {
                                if (expression instanceof NormalAnnotation) {
                                    final NormalAnnotation atAnnotation = (NormalAnnotation) expression;

                                    final AtData atDatum = inject.getAtData()[atIndex];
                                    remapAtAnnotation(ast, declaringClass, atAnnotation, atDatum);
                                }
                                atIndex++;
                            }
                        }
                        else if (pair.getValue() instanceof NormalAnnotation) {
                            final NormalAnnotation atAnnotation = (NormalAnnotation) pair.getValue();

                            final AtData atDatum = inject.getAtData()[atIndex];
                            remapAtAnnotation(ast, declaringClass, atAnnotation, atDatum);
                        }
                    }

                    // Remap @Slice
                    if (Objects.equals("slice", pair.getName().getIdentifier())) {
                        // it could be a SingleMemberAnnotation here but we don't care about that case

                        if (pair.getValue() instanceof ArrayInitializer) {
                            final ArrayInitializer value = (ArrayInitializer) pair.getValue();

                            for (final Object expression : value.expressions()) {
                                if (expression instanceof NormalAnnotation) {
                                    final NormalAnnotation atAnnotation = (NormalAnnotation) expression;

                                    final SliceData sliceDatum = inject.getSliceData()[sliceIndex];
                                    this.remapSliceAnnotation(ast, declaringClass, atAnnotation, sliceDatum);
                                }
                                sliceIndex++;
                            }
                        }
                        else if (pair.getValue() instanceof NormalAnnotation) {
                            final NormalAnnotation atAnnotation = (NormalAnnotation) pair.getValue();

                            final SliceData sliceDatum = inject.getSliceData()[sliceIndex];
                            this.remapSliceAnnotation(ast, declaringClass, atAnnotation, sliceDatum);
                        }
                    }
                }
            }

            // TODO: A similar "should be refactored" case.
            //  However, this behaves closer to @At("INVOKE") and @At("FIELD"),
            //  and needs to reimplement At remapping instead.
            // MixinExtra's @Definition
            if (Objects.equals(DEFINITION, annotationType)) {
                Object[] rawMethod = null;
                Object[] rawField = null;

                for (final IMemberValuePairBinding pair : annotation.getDeclaredMemberValuePairs()) {
                    if (Objects.equals("method", pair.getName())) {
                        rawMethod = (Object[]) pair.getValue();
                    }
                    if (Objects.equals("field", pair.getName())) {
                        rawField = (Object[]) pair.getValue();
                    }
                }

                final NormalAnnotation originalAnnotation = (NormalAnnotation) node.modifiers().get(i);
                for (final Object raw : originalAnnotation.values()) {
                    final MemberValuePair pair = (MemberValuePair) raw;

                    // note: rawMethod and rawField should not be null here
                    if (Objects.equals("method", pair.getName().getIdentifier())) {
                        assert rawMethod != null;

                        remapAtLikeField(ast, rawMethod, pair.getValue(), "INVOKE");
                    }
                    else if (Objects.equals("field", pair.getName().getIdentifier())) {
                        assert rawField != null;

                        remapAtLikeField(ast, rawField, pair.getValue(), "FIELD");
                    }
                }
            }

            // TODO: This should be refactored such that InjectData can handle this case,
            //  or other arbitrary cases like it, instead of being a bodge on.
            //  This largely duplicates the previous branch while specialising it.
            // MixinSquared's @TargetHandle
            // This is going to be a rare case, but might as well be comprehensive.
            if (Objects.equals(TARGET_HANDLER, annotationType)) {
                // Special casing as it uses `name` instead of `method`.
                final InjectTarget[] inject = getInjectTargets(annotation, "name");

                final String[] injectTargets = new String[inject.length];
                for (int j = 0; j < inject.length; j++) {
                    final InjectTarget injectTarget = inject[j];
                    injectTargets[j] = remapInjectTarget(target, injectTarget, binding);
                }

                final NormalAnnotation originalAnnotation = (NormalAnnotation) node.modifiers().get(i);
                for (final Object raw : originalAnnotation.values()) {
                    final MemberValuePair pair = (MemberValuePair) raw;

                    // Remap the method pair, like above.
                    if (Objects.equals("name", pair.getName().getIdentifier())) {
                        remapMethod(ast, pair, injectTargets);
                    }
                }
            }
        }

        return true;
    }

    // TODO: refactor InjectInfo to not require this out of band case.
    private static InjectTarget[] getInjectTargets(final IAnnotationBinding binding, final String method) {
        InjectTarget[] injectTargets = {};

        for (final IMemberValuePairBinding pair : binding.getDeclaredMemberValuePairs()) {
            if (Objects.equals(method, pair.getName())) {
                final Object[] raw = (Object[]) pair.getValue();

                injectTargets = new InjectTarget[raw.length];
                for (int i = 0; i < raw.length; i++) {
                    injectTargets[i] = InjectTarget.of((String) raw[i]);
                }
            }
        }

        return injectTargets;
    }

    // TODO: Make this not a bodge. The relevant consumer doesn't need the injection point.
    private static AtData[] getAtDataArray(final String injectionPoint, final String[] array) {
        final AtData[] atData = new AtData[array.length];

        for (int i = 0; i < array.length; i++) {
            atData[i] = AtData.from(injectionPoint, array[i], null);
        }

        return atData;
    }

    private void remapMethod(final AST ast, final MemberValuePair pair, final String[] injectTargets) {
        if (pair.getValue() instanceof StringLiteral || pair.getValue() instanceof InfixExpression) {
            replaceExpression(ast, this.context, pair.getValue(), injectTargets[0]);
        }
        else if (pair.getValue() instanceof ArrayInitializer) {
            final ArrayInitializer array = (ArrayInitializer) pair.getValue();
            for (int j = 0; j < array.expressions().size(); j++) {
                final StringLiteral original = (StringLiteral) array.expressions().get(j);
                replaceExpression(ast, this.context, original, injectTargets[j]);
            }
        }
    }

    private String remapInjectTarget(final ClassMapping<?, ?> target, final InjectTarget injectTarget, final IMethodBinding binding) {
        final String targetName = injectTarget.getTargetName();

        if (injectTarget.getFieldType().isPresent()) {
            // this is targeting a field
            final Type fieldType = injectTarget.getFieldType().get();

            for (final FieldMapping mapping : target.getFieldMappings()) {
                if (Objects.equals(targetName, mapping.getObfuscatedName())) {
                    if (mapping.getType().isPresent() && !Objects.equals(mapping.getType().get(), fieldType)) {
                        // the mapping has a type but it is different than the target type
                        continue;
                    }

                    final FieldSignature deobfuscatedSignature = mapping.getDeobfuscatedSignature();
                    String deobfuscatedFieldType = deobfuscatedSignature.getType()
                            .map(FieldType::toString)
                            .orElse(null);
                    if (deobfuscatedFieldType == null) {
                        deobfuscatedFieldType = this.mappings.deobfuscate(fieldType).toString();
                    }

                    return deobfuscatedFieldType != null ?
                            deobfuscatedSignature.getName() + ":" + deobfuscatedFieldType :
                            deobfuscatedSignature.getName();
                }
            }
        }
        else {
            // Handle cases where there are multiple matching obfuscated methods and only one valid deobfuscated target
            MethodDescriptor methodDescriptor = injectTarget.getMethodDescriptor().orElse(null);
            boolean needsDescriptor = shouldCalculateDescriptor(target, targetName);
            List<ITypeBinding> targetMethodParams = null;
            if (methodDescriptor == null && binding != null && needsDescriptor) {
                List<ITypeBinding> bindingParams = List.of(binding.getParameterTypes());
                int ciIndex = bindingParams.stream().filter(t -> CALLBACK_TYPES.contains(t.getBinaryName())).findFirst().map(bindingParams::indexOf).orElse(-1);
                if (ciIndex != -1) {
                    ITypeBinding returnType = bindingParams.get(ciIndex);
                    String methodParams = bindingParams.subList(0, ciIndex).stream().map(MixinRemapperVisitor::getTypeDescriptor).collect(Collectors.joining(""));
                    boolean isVoid = returnType.getBinaryName().equals(CALLBACK_INFO);
                    // Do we know the return type?
                    if (isVoid || returnType.getTypeParameters().length > 0) {
                        String returnTypeDesc = isVoid ? "V" : getTypeDescriptor(returnType.getTypeParameters()[0]);
                        methodDescriptor = MethodDescriptor.of("(" + methodParams + ")" + returnTypeDesc);   
                    }
                    // Try to match based on method params
                    else {
                        targetMethodParams = bindingParams.subList(0, bindingParams.size() - 1);
                    }
                }
            }

            // this is probably targeting a method
            for (final MethodMapping mapping : target.getMethodMappings()) {
                if (Objects.equals(targetName, mapping.getObfuscatedName()) && matchDescriptor(methodDescriptor, needsDescriptor, mapping, targetMethodParams)) {
                    final MethodSignature deobfuscatedSignature = mapping.getDeobfuscatedSignature();

                    return shouldIncludeDescriptor(target, mapping, injectTarget.getMethodDescriptor()) ?
                            deobfuscatedSignature.getName() + deobfuscatedSignature.getDescriptor().toString() :
                            deobfuscatedSignature.getName();
                }
            }
        }

        final MappingSet mappings = target.getMappings();
        final String targetOwner = injectTarget.getOwnerName();
        final MethodDescriptor descriptor = injectTarget.getMethodDescriptor().orElse(null);
        final Type type = injectTarget.getFieldType().orElse(null);

        final StringBuilder remappedFull = new StringBuilder();
        if (targetOwner != null) {
            remappedFull.append("L").append(targetOwner).append(";");
        }
        remappedFull.append(targetName);
        if (descriptor != null) {
            remappedFull.append(mappings.deobfuscate(descriptor));
        }
        if (type != null) {
            remappedFull.append(':');
            remappedFull.append(mappings.deobfuscate(type));
        }

        return remappedFull.toString();
    }

    private boolean matchDescriptor(MethodDescriptor methodDescriptor, boolean needsDescriptor, MethodMapping mapping, List<ITypeBinding> targetMethodParams) {
        return methodDescriptor == null && (!needsDescriptor || matchParameters(mapping, targetMethodParams)) || methodDescriptor != null && methodDescriptor.equals(mapping.getDescriptor());
    }

    private boolean matchParameters(MethodMapping mapping, List<ITypeBinding> mixinParams) {
        if (mixinParams != null) {
            List<FieldType> targetParams = MethodDescriptor.of(mapping.getObfuscatedDescriptor()).getParamTypes();
            if (targetParams.size() == mixinParams.size()) {
                for (int i = 0; i < targetParams.size(); i++) {
                    if (!targetParams.get(i).toString().equals(getTypeDescriptor(mixinParams.get(i)))) {
                        return false;
                    }
                }
                return true;
            }
        }
        return false;
    }

    private boolean shouldIncludeDescriptor(ClassMapping<?, ?> target, MethodMapping mapping, Optional<MethodDescriptor> descriptor) {
        return descriptor.isPresent()
            || target.getMethodMappings().stream().map(Mapping::getDeobfuscatedName).filter(mapping.getDeobfuscatedName()::equals).count() > 1 
            && target.getMethodMappings().stream().map(Mapping::getObfuscatedName).filter(mapping.getObfuscatedName()::equals).count() == 1;
    }

    private boolean shouldCalculateDescriptor(ClassMapping<?, ?> target, String targetName) {
        // Calculate descriptor when there's multiple candidates with varying deobf names
        List<MethodMapping> candidates = target.getMethodMappings().stream().filter(m -> Objects.equals(targetName, m.getObfuscatedName())).toList();
        return candidates.size() > 1 && candidates.stream().anyMatch(m -> !Objects.equals(candidates.get(0).getDeobfuscatedName(), m.getDeobfuscatedName()));
    }

    private void remapSliceAnnotation(final AST ast, final ITypeBinding declaringClass, final NormalAnnotation atAnnotation,
                                      final SliceData sliceDatum) {
        for (final Object raw : atAnnotation.values()) {
            // this will always be a MemberValuePair
            final MemberValuePair pairRaw = (MemberValuePair) raw;

            // it could be a SingleMemberAnnotation here but we don't care about that case
            if (!(pairRaw.getValue() instanceof NormalAnnotation)) continue;

            if (Objects.equals("from", pairRaw.getName().getIdentifier())) {
                this.remapAtAnnotation(ast, declaringClass, (NormalAnnotation) pairRaw.getValue(), sliceDatum.getFrom());
            }
            if (Objects.equals("to", pairRaw.getName().getIdentifier())) {
                this.remapAtAnnotation(ast, declaringClass, (NormalAnnotation) pairRaw.getValue(), sliceDatum.getTo());
            }
        }
    }

    private void remapAtAnnotation(final AST ast, final ITypeBinding declaringClass, final NormalAnnotation atAnnotation, final AtData atDatum) {
        this.remapAtAnnotation(ast, declaringClass, atAnnotation, atDatum, "target", "desc");
    }

    private void remapAtAnnotation(final AST ast, final ITypeBinding declaringClass, final NormalAnnotation atAnnotation, final AtData atDatum, final String targetName, final String descName) {
        for (final Object atRaw : atAnnotation.values()) {
            // this will always be a MemberValuePair
            final MemberValuePair atRawPair = (MemberValuePair) atRaw;

            // check for the target
            if (Objects.equals(targetName, atRawPair.getName().getIdentifier())) {
                remapAtAnnotation(ast, atDatum, atRawPair.getValue());
            }

            // modern style @Desc
            if (Objects.equals(descName, atRawPair.getName().getIdentifier())) {
                if (!atDatum.getDesc().isPresent()) continue;
                this.remapDescAnnotation(ast, declaringClass, (Annotation) atRawPair.getValue(), atDatum.getDesc().get());
            }
        }
    }

    private void remapAtAnnotation(final AST ast, final AtData atDatum, Expression originalTarget) {
        // make sure everything is present
        if (atDatum.getClassName().isPresent()) {
            final String className = atDatum.getClassName().get();

            // get the class mapping of the class that owns the target we're remapping
            final ClassMapping<?, ?> atTargetMappings = this.mappings.computeClassMapping(className).orElse(null);
            if (atTargetMappings == null) {
                return;
            }

            final String deobfTargetClass = atTargetMappings.getFullDeobfuscatedName();

            if (atDatum.getTarget().isPresent()) {
                final InjectTarget atTarget = atDatum.getTarget().get();
                final String newTarget = remapInjectTarget(atTargetMappings, atTarget, null);
                String deobfTarget = "L" + deobfTargetClass + ";" + newTarget;
                replaceExpression(ast, this.context, originalTarget, deobfTarget);
            }
            else {
                // it's just the class name
                replaceExpression(ast, this.context, originalTarget, deobfTargetClass);
            }
        } else if (atDatum.getTarget().isPresent()) {
            atDatum.getTarget().get().getMethodDescriptor()
                    .ifPresent(desc -> replaceExpression(ast, this.context, originalTarget, this.mappings.deobfuscate(desc).toString()));
        }
    }

    // TODO: this could be refactored so that the sink doesn't need AtDesc and can just be fed the value directly.
    private void remapAtLikeField(final AST ast, final Object[] rawValues, final Expression expression, final String atInjectionPoint) {
        if (expression instanceof ArrayInitializer) {
            final ArrayInitializer value = (ArrayInitializer) expression;
            final List<?> expressions = value.expressions();

            for (int i = 0; i < expressions.size(); i++) {
                final AtData atDatum = AtData.from(atInjectionPoint, (String)rawValues[i], null);
                final Object expr = expressions.get(i);
                // Assume Expression for now. This is probably correct.
                remapAtAnnotation(ast, atDatum, (Expression) expr);
            }
        }
        else {
            final AtData atDatum = AtData.from(atInjectionPoint, (String)rawValues[0], null);
            remapAtAnnotation(ast, atDatum, expression);
        }
    }

    private void remapDescAnnotation(final AST ast, final ITypeBinding declaringClass,
                                     final Annotation annotation, final DescData descData) {
        // todo: be passed this information
        final MixinClass mixin = MixinClass.fetch(declaringClass, this.mappings);
        if (mixin == null) return;

        final ClassMapping<?, ?> owner = descData.getOwnerBinding() == null ?
                this.mappings.getClassMapping(mixin.getTargetNames()[0]).orElse(null) :
                this.mappings.getClassMapping(descData.getOwnerBinding().getBinaryName()).orElse(null);
        if (owner == null) return;

        final Type returnType = descData.getReturnBinding() == null ?
                VoidType.INSTANCE :
                BombeBindings.convertType(descData.getReturnBinding());

        // try remap as a method
        {
            final List<FieldType> arguments = new ArrayList<>(descData.getArgBindings().length);

            for (final ITypeBinding argBinding : descData.getArgBindings()) {
                arguments.add((FieldType) BombeBindings.convertType(argBinding));
            }

            final MethodMapping methodMapping = owner.getMethodMapping(new MethodSignature(
                    descData.getName(),
                    new MethodDescriptor(arguments, returnType)
            )).orElse(null);
            if (methodMapping != null) {
                replaceValueInAnnotation(ast, this.context, annotation, methodMapping.getDeobfuscatedName());
                return;
            }
        }

        // try remap as a field
        if (descData.getArgBindings().length == 0 && returnType != VoidType.INSTANCE) {
            // TODO: implement
        }
    }

    private void visit(final SimpleName node, final IBinding binding) {
        switch (binding.getKind()) {
            case IBinding.VARIABLE:
                this.remapField(node, ((IVariableBinding) binding).getVariableDeclaration());
                break;
        }
    }

    @Override
    public final boolean visit(final SimpleName node) {
        final IBinding binding = node.resolveBinding();
        if (binding != null) {
            this.visit(node, binding);
        }
        return false;
    }

    @Override
    public boolean visit(final TypeDeclaration node) {
        this.remapPrivateMixinTarget(node.getAST(), node, node.resolveBinding());
        return true;
    }

    private static void replaceExpression(final AST ast, final RewriteContext context, final Expression original, final String replacement) {
        final StringLiteral replacementLiteral = ast.newStringLiteral();
        replacementLiteral.setLiteralValue(replacement);
        context.createASTRewrite().replace(original, replacementLiteral, null);
    }

    private static void replaceValueInAnnotation(final AST ast, final RewriteContext context, final Annotation rawAnnotation, final String replacement) {
        if (rawAnnotation.isNormalAnnotation()) {
            final NormalAnnotation annotationNode = (NormalAnnotation) rawAnnotation;

            for (final Object raw : annotationNode.values()) {
                final MemberValuePair pair = (MemberValuePair) raw;

                // Remap the method pair
                if (Objects.equals("value", pair.getName().getIdentifier())) {
                    final StringLiteral original = (StringLiteral) pair.getValue();
                    replaceExpression(ast, context, original, replacement);
                }
            }
        }
        else if (rawAnnotation.isSingleMemberAnnotation()) {
            final SingleMemberAnnotation annotationNode = (SingleMemberAnnotation) rawAnnotation;
            final StringLiteral original = (StringLiteral) annotationNode.getValue();
            replaceExpression(ast, context, original, replacement);
        }
        else if (rawAnnotation.isMarkerAnnotation()) {
            // FIXME: Look at this properly
        }
        else {
            throw new RuntimeException("Unexpected annotation: " + rawAnnotation.getClass().getName());
        }
    }

    private static FieldSignature convertSignature(final String name, final ITypeBinding type) {
        return new FieldSignature(name, (FieldType) convertType(type));
    }

    private static MethodSignature convertSignature(final String name, final IMethodBinding binding) {
        final ITypeBinding[] parameterBindings = binding.getParameterTypes();
        final List<FieldType> parameters = new ArrayList<>(parameterBindings.length);

        for (final ITypeBinding parameterBinding : parameterBindings) {
            parameters.add((FieldType) convertType(parameterBinding));
        }

        return new MethodSignature(name, new MethodDescriptor(parameters, convertType(binding.getReturnType())));
    }

    private static String getTypeDescriptor(ITypeBinding binding) {
        return binding.isPrimitive() ? binding.getBinaryName() : "L" + binding.getBinaryName().replace('.', '/') + ";";
    }
}
